/***************************************************************************
 *
 * Copyright (C) 2001 International Business Machines
 * All rights reserved.
 *
 * This file is part of the GPFS mmfslinux kernel module.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer. 
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written
 *     permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *************************************************************************** */
/*
 * Abstraction of an I/O buffer, Linux implementation
 *
 * Contents:
 *   struct cxiKernelIOBufferDesc_t
 *   struct cxiIOBufferAttachment_t
 *   InitBufferAttachment
 *   struct cxiContiguousBuffer_t
 *   InitContiguousBuffer
 *   EXTERNC int kxPinKernelIOBuffer
 *   Methods for manipulating cxiIOBuffer_t's
 *   #define __CXI_BUFFERS_ARE_CONTIGUOUS
 *   GetDiskInfoX
 */

/* $Id: cxiIOBuffer-plat.h,v 1.20 2001/09/22 20:10:33 dcraft Exp $
 *
 * $Log: cxiIOBuffer-plat.h,v $
 * Revision 1.20  2001/09/22 20:10:33  dcraft
 * Remove kiobufs from cxiKernelIODescriptor_t.  Use temporary
 * kiobufs for map/unmap.   Remove dead code and dead comments
 * in portability layer and update readmes and license.
 * Fix traceback to appear in mmfs.log file.
 *
 * Revision 1.19  2001/09/10 16:22:02  wyllie
 * Increase number of pages mapped by one cxiKernelIOBufferDesc_t so that
 * the I/O code does not have to worry about spanning cxiKernelIOBufferDesc_ts.
 *
 * Revision 1.18  2001/08/24 13:11:45  gjertsen
 * Remove bogus address for IA64 LINUX_PAGE_POOL_BASE.
 *
 * Revision 1.17  2001/07/13 19:56:42  wyllie
 * Get PAGE_OFFSET by asking the mmfslinux_... kernel module rather than
 * compiling the constant into proprietary code.  Rename constants needed to
 * determine the GPFS memory map and export them to non-proprietary files.
 *
 * Revision 1.16  2001/06/25 21:23:17  tee
 * Make direct I/O work on Linux.
 *
 * Revision 1.15  2001/06/04 21:23:40  eshel
 * Allocate 256K stacks for pthreads with one guard page at the end.
 *
 * Revision 1.14  2001/04/10 21:11:34  wyllie
 * Convert cxiIOBuffer.C from C++ to C.
 *
 * Revision 1.13  2001/04/02 22:12:11  wyllie
 * Prefix platform-dependent symbols describing the layout of shared memory
 * with the platform name.
 *
 * Revision 1.12  2001/01/08 20:49:03  gjertsen
 * Minor code cleanup. No functional changes.
 *
 * Revision 1.11  2000/12/15 13:57:19  gjertsen
 * Clean up documentation.
 *
 * Revision 1.10  2000/12/14 21:58:26  wyllie
 * Remove KIOBUF_STUFF ifdefs
 *
 * Revision 1.9  2000/11/09 22:00:07  wyllie
 * Increase size of virtual area assigned to page pool to 960M to have enough
 * sparseness to allow a 512M page pool.  Define the address of the TM malloc
 * pool.
 *
 * Revision 1.8  2000/10/26  20:55:19  gjertsen
 * Purge out ugly USE_CWRAPPERS and export module symbols explicitly
 * as the default (in IA64 safe manner).
 *
 * Revision 1.7  2000/09/28  19:43:08  eshel
 * Change core dump to include the shared segment.
 *
 * Revision 1.6  2000/08/29  18:33:10  dcraft
 * Header include cleanup.
 *
 * Revision 1.5  2000/08/28  14:15:25  gjertsen
 * Use C wrapper functions for C++ class methods in
 * cxi interface. Convert all cxi functions to C interface.
 *
 * Revision 1.4  2000/08/21  22:16:11  dcraft
 * Create cxiDev_t type that is based on user level dev_t.  Provide
 * mapping functions between kernel, user, and inode device field.
 * Fix NLS yes/no query.
 *
 * Revision 1.3  2000/08/10  00:02:58  wyllie
 * Buffer management for Linux, phase III: physical disk I/O.  I/O is done
 * synchronously by kxStartIO.  For well-aligned buffers (full blocks starting
 * at offset 0), uses the kiobufs already built.  For other I/O, builds a
 * temporary kiobuf to pass to brw_kiovec.
 *
 * Revision 1.2  2000/08/01  17:08:20  wyllie
 * Buffer management for Linux, phase II: for each Buffer created by the page
 * pool manager, create a shadow of the buffer in the kernel that contains
 * kiobufs pointing to the Linux struct page objects for each page in the data
 * area of the Buffer.  Use these mappings to implement uXfer, kXfer, etc.
 * Not yet fully functional; requires a -D flag to activate.
 *
 * Revision 1.1  2000/06/30  16:24:17  wyllie
 * Buffer management for Linux, phase I: abstract kernel mapping of page pool
 * objects into the cxiIOBuffer_t class.  Define interfaces for manipulating
 * such I/O buffers from the kernel.  Change all kernel users of Buffer objects
 * to use the new interfaces, which are fully implemented for AIX and stubbed
 * out for Linux.
 *
 */

#ifndef _h_cxiIOBuffer_plat
#define _h_cxiIOBuffer_plat

/* Address of the first byte past the end of memory addressible by
   processes (PAGE_OFFSET), and routine to get this value from the kernel.
   Stacks are below this address. */
EXTERNC UIntPtr KernelBoundary;
EXTERNC int kxGetKernelBoundary(UIntPtr* kBoundP);

/* User address space range used for page pool. */
#ifdef GPFS_ARCH_I386
#define LINUX_PAGE_POOL_BASE  0x44000000
#endif
#ifdef GPFS_ARCH_IA64
/* Use shared memory region after TM pool (offset by chunk size) */
#define LINUX_PAGE_POOL_BASE  0x2000000024000000
#endif
EXTERNC char* LinuxPoolUpperBoundary;
#define POOL_MMAP_CHUNK_SIZE  0x04000000

/* Address where token manager malloc pool begins */
#ifdef GPFS_ARCH_I386
# define TM_POOL_START 0x30000000
#endif
#ifdef GPFS_ARCH_IA64
# define TM_POOL_START 0x2000000010000000
#endif


/* Buffers in user address space must be aligned to a boundary of this size
   in order to perform an I/O request. */
#define IOBUF_ALIGN_SIZE PAGE_SIZE

/* Kernel data structure associated with an I/O buffer.  I/O buffers that
   are pinned (or attached) point to one of these structures from their
   kernelIOBufferDescP field.  It describes the physical pages occupied by
   the I/O buffer using Linux kiobufs.  These are linked together in a
   global list anchored in the kernel so that pinned storage can be released
   when the GPFS daemon terminates abnormally.  Each I/O buffer has one
   cxiKernelIOBufferDesc_t on this global list.  However, since one
   cxiKernelIOBufferDesc_t can map at most PAGES_PER_KIBD pages, large I/O
   buffers require multiple cxiKernelIOBufferDesc_t's.  */
struct cxiKernelIOBufferDesc_t
{
  /* Daemon address for beginning of I/O buffer.  This address must be
     aligned on a page boundary. */
  char* kibdVaddr;

  /* Number of pages described by this cxiKernelIOBufferDesc_t. */
  int kibdPages;

  /* Number of pages described by this chain of cxiKernelIOBufferDesc_t 
   * Only valid for the first cxiKernelIOBufferDesc_t in the chain.
   */
  int kibdTotalPages;

  /* List pointer.  Used for a chain of cxiKernelIOBufferDesc_t's. */
  struct cxiKernelIOBufferDesc_t* kibdNextP;

  /* An I/O buffer is described by a chain of cxiKernelIOBufferDesc_t,
   * of which the head descriptor is placed on a global list.   Thus these
   * fields are only valid for the first cxiKernelIOBufferDesc_t in the 
   * chain of descriptors.
   */
  struct cxiKernelIOBufferDesc_t* gblNextP;
  struct cxiKernelIOBufferDesc_t* gblPrevP;

#define PAGES_PER_KIBD (64*1024/PAGE_SIZE)  /* 64K */
  char* maplist[PAGES_PER_KIBD];
};

/* Initialization and termination routines.  Called at module load
   and unload, respectively. */
EXTERNC void KibdModuleInit();
EXTERNC void KibdModuleTerm();

/* Create a cxiKernelIOBufferDesc_t object (or list of cxiKernelIOBufferDesc_t
   objects) describing an I/O buffer in the user address space of the
   calling process and link it onto the list of all such objects.  Pins
   the user-level buffer.  The buffer virtual address must be on a page
   boundary.  The length can be arbitrarily large, but must be a multiple
   of the page size.  Returns 0 if successful, non-zero if unsuccessful.
   */
EXTERNC int cxiKibdPin(char* vaddr, int len,
                       struct cxiKernelIOBufferDesc_t** kibdPP);

/* Remove a cxiKernelIOBufferDesc_t object from the list of all
   such objects, destroy it and all chained cxiKernelIOBufferDesc_t objects
   associated with it, and unpin the associated user-level buffer. */
EXTERNC void cxiKibdUnpin(struct cxiKernelIOBufferDesc_t* kibdP);

/* Free all cxiKernelIOBufferDesc_t's, and unpin their underlying storage. */
EXTERNC void cxiKibdUnpinAll();

/* Given a cxiKernelIOBufferDesc_t *kibdP mapping an area of memory, split
   it into two pieces.  The first frontPages pages will belong to *frontPP,
   and the excess pages will be assigned to *rearPP.  frontPages must be
   strictly less the initial length of *kibdP.  kibdP may be reused as
   either *frontPP or *rearPP, so it should not be used after a successful
   return from this routine.  If this routine fails (rc not EOK), then
   nothing is known about the state of the orginal *kibdP or either
   **frontPP or **rearPP. */
EXTERNC int cxiKibdSplit(struct cxiKernelIOBufferDesc_t* kibdP, int frontPages,
                         struct cxiKernelIOBufferDesc_t** frontPP,
                         struct cxiKernelIOBufferDesc_t** rearPP);

/* Combine two cxiKernelIOBufferDesc_t's *frontP and *rearP that map
   contiguous buffers into a new cxiKernelIOBufferDesc_t **mergedPP.  The
   *frontP and *rearP objects may be reused within **mergedPP, so they
   should not be used after a successful return from this routine.  If
   this routine fails (rc not EOK), then nothing is known about the state
   of the orginal *frontP or *rearP or **mergedPP.*/
EXTERNC int cxiKibdMerge(struct cxiKernelIOBufferDesc_t* frontP,
                         struct cxiKernelIOBufferDesc_t* rearP,
                         struct cxiKernelIOBufferDesc_t** mergedPP);


/* Handle that describes a particular cxiIOBuffer_t that has been attached.
   On Linux, this is a pointer to a cxiLinuxKernelIOBufferDesc_t. */
struct cxiIOBufferAttachment_t
{
  struct cxiKernelIOBufferDesc_t* kDescP;
};


/* Initialize a cxiIOBufferAttachment_t */
static inline void InitBufferAttachment(struct cxiIOBufferAttachment_t* baP)
{
  baP->kDescP = NULL;
};



/* Result of making a read-only copy of a portion of an I/O buffer.  On
   Linux, this must record the base address of the copy buffer, if one was
   required.  If data was mapped in place, the cxiContiguousBuffer_t records
   which page was kmapped. */
struct cxiContiguousBuffer_t
{
  /* Base of storage allocated with cxiMallocPinned, or NULL if data is
     referenced in place. */
  char* mallocedBaseP;

  /* Pointer used to remember which page to unmap, or NULL if data was copied
     to mallocedBaseP by mapContiguousRO. */
  void* pageP;
};


/* Initialize a cxiContiguousBuffer_t */
static inline void InitContiguousBuffer(struct cxiContiguousBuffer_t* cbP)
{
  cbP->mallocedBaseP = NULL;
  cbP->pageP = NULL;
}


/* Kernel calls used by cxiK... routines to call the Kibd... routines */
EXTERNC int kxPinKernelIOBuffer(char* vaddr, int len,
                                struct cxiKernelIOBufferDesc_t** pinnedPP);
EXTERNC int kxUnpinKernelIOBuffer(struct cxiKernelIOBufferDesc_t* pinnedP);
EXTERNC int kxSplitKernelIOBuffer(struct cxiKernelIOBufferDesc_t* origP,
                                  int frontPages,
                                  struct cxiKernelIOBufferDesc_t** frontPP,
                                  struct cxiKernelIOBufferDesc_t** rearPP);
EXTERNC int kxMergeKernelIOBuffers(struct cxiKernelIOBufferDesc_t* frontP,
                                   struct cxiKernelIOBufferDesc_t* rearP,
                                   struct cxiKernelIOBufferDesc_t** mergedPP);
EXTERNC int kxUnpinAllKernelIOBuffers();


/* Methods for manipulating cxiIOBuffer_t's */

/* Return true if the fields describing the IOBuffer are self-consistent */
static inline Boolean IOBufferIsConsistent(struct cxiIOBuffer_t* iobP)
  { return true; };

/* Pin the pages belonging to this I/O buffer */
EXTERNC void KPinIOBuffer(struct cxiIOBuffer_t* iobP);

/* Unpin the pages belonging to this I/O buffer */
EXTERNC void KUnpinIOBuffer(struct cxiIOBuffer_t* iobP);

/* Split the kernel buffer descriptor into two adjacent I/O buffers */
EXTERNC void KSplitIOBuffer(struct cxiIOBuffer_t* iobP, int frontPages,
                            struct cxiIOBuffer_t* rearBufP);

/* Merge the kernel buffer descriptors of two adjacent I/O buffers.  The
   I/O buffer p should be destroyed after this call, since its pages will
   be merged into the buffer *iobP. */
EXTERNC void KMergeIOBuffer(struct cxiIOBuffer_t* iobP, struct cxiIOBuffer_t* p);

/* Read or write the given sectors from dev.  Data should be placed into
   the I/O buffer beginning at byte offset bufOffset.  Returns EOK
   on success, negative values on error.  All of the data to be
   transferred will be in the first cxiKernelIOBufferDesc_t. */
EXTERNC int cxiKDoIO(struct cxiKernelIOBufferDesc_t* kibdP,
                     Boolean isWrite, cxiDev_t dev, Int32 startSector,
                     int nSectors, int sectorSize, int bufOffset);


/* On Linux, I/O buffers can be accessed at contiguous virtual addresses
   from the daemon process, but not from kernel code */
#ifndef _KERNEL
#define __CXI_BUFFERS_ARE_CONTIGUOUS
#endif

/* Routine to set up the disk block size and get disk parameters */
EXTERNC int GetDiskInfoX(cxiDev_t devId, struct cxiDiskInfo_t* diskInfoP);

#endif  /* _h_cxiIOBuffer_plat */

